document.addEventListener("DOMContentLoaded", function() {
    const gridContainer = document.getElementById('sudoku-grid');
    const generateButton = document.getElementById('generate');
    const solveButton = document.getElementById('solve');
    const answerButton = document.getElementById('answer');
    const messageElement = document.createElement('div');
    messageElement.className = 'message';
    document.body.appendChild(messageElement);

    let solution = [];

    // Generate the Sudoku grid
    function createGrid(sudoku) {
        gridContainer.innerHTML = ''; // Clear previous grid
        for (let i = 0; i < 81; i++) {
            const cell = document.createElement('input');
            cell.type = 'text';
            cell.maxLength = 1;
            cell.className = 'cell';
            cell.value = sudoku[Math.floor(i / 9)][i % 9] !== 0 ? sudoku[Math.floor(i / 9)][i % 9] : '';
            cell.readOnly = false;  // Make the cells editable by the user
            gridContainer.appendChild(cell);
        }
    }

    // Generate a Sudoku puzzle with random numbers as answers
    function generateRandomPuzzle() {
        const grid = Array.from({ length: 9 }, () => Array(9).fill(0));
        
        for (let i = 0; i < 9; i++) {
            for (let j = 0; j < 9; j++) {
                grid[i][j] = Math.floor(Math.random() * 9) + 1; // Filling with random numbers (1-9)
            }
        }

        // Clear random cells to make it a puzzle
        clearCells(grid);

        solution = grid; // Set the solution to this random grid
        createGrid(grid);
    }

    // Clear random cells to generate a puzzle
    function clearCells(grid) {
        const cellsToClear = 40; // Number of cells to clear
        let count = 0;
        while (count < cellsToClear) {
            const i = Math.floor(Math.random() * 9);
            const j = Math.floor(Math.random() * 9);
            if (grid[i][j] !== 0) {
                grid[i][j] = 0; // Clear the cell
                count++;
            }
        }
    }

    answerButton.addEventListener('click', function() {
        if (solution.length > 0) {
            const cells = document.querySelectorAll('.cell');
            solution.forEach((row, i) => {
                row.forEach((num, j) => {
                    cells[i * 9 + j].value = num;
                });
            });
            messageElement.textContent = "Answers displayed. Check your puzzle!";
            
            // Hide the message after 2 seconds
            setTimeout(() => {
                messageElement.textContent = ""; // Clear the message after 2 seconds
            }, 2000);
        }
    });

    // Show solved or not solved message based on the user input
    solveButton.addEventListener('click', function() {
        const cells = document.querySelectorAll('.cell');
        let solved = true;

        cells.forEach((cell, index) => {
            const correctValue = solution[Math.floor(index / 9)][index % 9];
            if (cell.value.trim() == '') {
                solved = false; // If any cell is empty in the user's answer
            } else if (parseInt(cell.value) !== correctValue) {
                solved = false; // If any user's input doesn't match the solution
            }
        });

        if (solved) {
            messageElement.textContent = "You did it! The puzzle has been solved.";
        } else {
            messageElement.textContent = "Puzzle not solved! Check your inputs.";
        }

        // Hide the message after 2 seconds
        setTimeout(() => {
            messageElement.textContent = ""; // Clear the message after 2 seconds
        }, 2000);
    });

    // Generate Sudoku on button click
    generateButton.addEventListener('click', generateRandomPuzzle);
});